<?php

namespace App\Http\Controllers\Admin;

use App\Models\Deposit;
use App\Http\Controllers\Controller;
use App\Models\General;
use App\Models\InvestLog;
use App\Models\Kyc;
use App\Models\Wallet;
use App\Models\Plan;
use App\Models\Ranking;
use App\Models\Referral;
use App\Models\Transaction;
use App\Models\User;
use App\Models\UserRanking;
use App\Models\WithdrawLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class AdminController extends Controller
{
    protected function totalEarn($investLog)
    {
        if (count($investLog) > 0) {
            $totalEarn = 0;
            foreach ($investLog as $value) {
                $totalEarn += ((floatval($value->invest_amount) * floatval($value->get_percent)) / 100) * $value->took_action;
            }
            $total_earn = round($totalEarn, 2);
        } else {
            $total_earn = 0;
        }
        return $total_earn;
    }

    public function dayList()
    {
        $totalDays = cal_days_in_month(CAL_GREGORIAN, date('m'), date('Y'));
        $daysByMonth = [];
        for ($i = 1; $i <= $totalDays; $i++) {
            array_push($daysByMonth, ['Day ' . sprintf('%02d', $i) => 0]);
        }

        return collect($daysByMonth)->collapse();
    }

    public function adminIndex()
    {
        $data['userRecord'] = collect(
            User::selectRaw('count(CASE WHEN status = 1  THEN id END) AS activeUser')
                ->selectRaw('COUNT((CASE WHEN created_at >= CURDATE()  THEN id END)) AS todayJoin')
                ->get()
                ->makeHidden(['name', 'mobile'])
                ->toArray(),
        )->collapse();

        $data['total_user'] = User::count();
        $data['total_user_mon'] = User::whereMonth('created_at', Carbon::now()->month)->count();

        $data['total_plan'] = plan::count();

        $investLogMonth = InvestLog::whereMonth('created_at', Carbon::now()->month)->get();
        $data['month_earn'] = $this->totalEarn($investLogMonth);

        $investLog = InvestLog::get();
        $data['total_earn'] = $this->totalEarn($investLog);

        $data['total_ac_user'] = User::where('status', 1)->count();
        $data['total_ac_user_month'] = User::where('status', 1)
            ->whereMonth('created_at', Carbon::now()->month)
            ->count();

        $data['total_bn_user'] = User::where('status', 0)->count();
        $data['total_bn_user_month'] = User::where('status', 0)
            ->whereMonth('created_at', Carbon::now()->month)
            ->count();

        $data['total_deposit'] = Deposit::where('status', 1)->sum('amount');
        $data['today_deposit'] = Deposit::where('created_at', Carbon::now())->where('status', 1)->sum('amount');
        $data['total_deposit_month'] = Deposit::where('status', 1)
            ->whereMonth('updated_at', Carbon::now()->month)
            ->sum('amount');

        $data['total_withdraw'] = WithdrawLog::where('status', 1)->sum('amount');
        $data['today_withdraw'] = WithdrawLog::where('created_at', Carbon::now())->where('status', 1)->sum('amount');
        $data['panding_withdraw'] = WithdrawLog::where('status', 0)->sum('id');
        $data['reject_withdraw'] = WithdrawLog::where('status', 2)->sum('id');
        $data['total_withdraw_month'] = WithdrawLog::where('status', 1)
            ->whereMonth('updated_at', Carbon::now()->month)
            ->sum('amount');

        $data['total_invest'] = InvestLog::sum('invest_amount');
        $data['running_invest'] = InvestLog::where('status', 0)->sum('id');
        $data['complete_invest'] = InvestLog::where('status', 1)->sum('id');
        $data['today_invest'] = InvestLog::where('created_at', Carbon::now())->sum('id');
        $data['total_invest_month'] = InvestLog::whereMonth('created_at', Carbon::now()->month)->sum('invest_amount');

        $data['roi_plans'] = Plan::where('return_time_status', 1)->get();
        $data['fixed_plans'] = Plan::where('return_time_status', 0)->get();

        $data['investLog'] = InvestLog::latest('updated_at')->paginate();

        $data['latestUser'] = User::latest()->limit(5)->get();

        $page_title = 'Dashboard';

        $trxReport['date'] = collect([]);

        $trxReport['date'] = $trxReport['date']->unique()->toArray();

        $report['months'] = collect([]);
        $report['deposit_month_amount'] = collect([]);
        $report['withdraw_month_amount'] = collect([]);

        $depositsMonth = Deposit::where('created_at', '>=', Carbon::now()->subYear())->where('status', 1)->selectRaw('SUM(CASE WHEN status = 1 THEN amount END) as depositAmount')->selectRaw("DATE_FORMAT(created_at,'%M-%Y') as months")->orderBy('created_at')->groupBy('months')->get();

        $depositsMonth->map(function ($depositData) use ($report) {
            $report['months']->push($depositData->months);
            $report['deposit_month_amount']->push(getAmount($depositData->depositAmount));
        });

        $withdrawalMonth = WithdrawLog::where('created_at', '>=', Carbon::now()->subYear())->where('status', 1)->selectRaw('SUM( CASE WHEN status = 1 THEN amount END) as withdrawAmount')->selectRaw("DATE_FORMAT(created_at,'%M-%Y') as months")->orderBy('created_at')->groupBy('months')->get();
        $withdrawalMonth->map(function ($withdrawData) use ($report) {
            if (!in_array($withdrawData->months, $report['months']->toArray())) {
                $report['months']->push($withdrawData->months);
            }
            $report['withdraw_month_amount']->push(getAmount($withdrawData->withdrawAmount));
        });

        $months = $report['months'];

        for ($i = 0; $i < $months->count(); ++$i) {
            $monthVal = Carbon::parse($months[$i]);
            if (isset($months[$i + 1])) {
                $monthValNext = Carbon::parse($months[$i + 1]);
                if ($monthValNext < $monthVal) {
                    $temp = $months[$i];
                    $months[$i] = Carbon::parse($months[$i + 1])->format('F-Y');
                    $months[$i + 1] = Carbon::parse($temp)->format('F-Y');
                } else {
                    $months[$i] = Carbon::parse($months[$i])->format('F-Y');
                }
            }
        }

        $investMonth = Transaction::where('status', '0')->where('created_at', '>=', Carbon::now()->subYear())->selectRaw('SUM(amount) as amount')->selectRaw("DATE_FORMAT(created_at,'%M-%Y') as months")->orderBy('created_at')->groupBy('months')->get();
        $investMonth->map(function ($investData) use ($report) {
            $report['months']->push($investData->months);
        });

        $interestMonth = Transaction::where('status', '4')->where('created_at', '>=', Carbon::now()->subYear())->selectRaw('SUM(amount) as amount')->selectRaw("DATE_FORMAT(created_at,'%M-%Y') as months")->orderBy('created_at')->groupBy('months')->get();

        $interestMonth->map(function ($interestData) use ($report) {
            $report['months']->push($interestData->months);
        });

        return view('admin.home', $data, compact('page_title', 'report', 'months', 'depositsMonth', 'withdrawalMonth', 'trxReport', 'investMonth', 'interestMonth'));
    }

    public function kycPendingList()
    {
        $page_title = 'Pending KYC';
        $logs = KYC::orderBy('id', 'DESC')
            ->where('status', 0)
            ->with(['user'])
            ->paginate(15);
        return view('admin.user.kycList', compact('logs', 'page_title'));
    }
    public function kycList()
    {
        $page_title = 'KYC Log';
        $logs = Kyc::orderBy('id', 'DESC')->where('status', '!=', 0)->paginate(15);
        return view('admin.user.kycList', compact('logs', 'page_title'));
    }

    public function userKycHistory(User $user)
    {
        $page_title = $user->name . ' : KYC Log';
        $logs = Kyc::orderBy('id', 'DESC')
            ->where('user_id', $user->id)
            ->paginate(15);
        return view('admin.user.kycList', compact('logs', 'page_title'));
    }

    public function kycAction(Request $request, $id)
    {
        $request->validate([
            'id' => 'required',
            'status' => ['required', Rule::in(['1', '2'])],
        ]);
        $data = Kyc::where('id', $request->id)
            ->whereIn('status', [0])
            ->with('user')
            ->firstOrFail();
        $basic = General::first();

        if ($request->status == '1') {
            $data->status = 1;
            $data->update();
            $user = $data->user;
            if ($data->kyc_type == 'address-verification') {
                $user->address_verify = 2;
            } else {
                $user->identity_verify = 2;
            }
            $user->save();

            @send_email($user, 'KYC_APPROVED');

            return redirect()->back()->with('success', 'Approve Successfully');
        } elseif ($request->status == '2') {
            $data->status = 2;
            $data->update();

            $user = $data->user;
            if ($data->kyc_type == 'address-verification') {
                $user->address_verify = 3;
            } else {
                $user->identity_verify = 3;
            }
            $user->save();

            @send_email($user, 'KYC_REJECTED');

            return redirect()->back()->with('success', 'Reject Successfully');
        }
    }

    public function gnlSetting()
    {
        $page_title = 'Settings';
        return view('admin.general.settings', compact('page_title'));
    }

    public function selTemplate()
    {
        $page_title = 'Template Settings';
        return view('admin.general.template', compact('page_title'));
    }

    public function adminLogout()
    {
        Auth::guard('admin')->logout();
        return redirect()->route('admin.login')->with('success', 'Logout successfully');
    }

    public function usersIndex()
    {
        $page_title = 'Users Management';
        $user = User::latest('id')->paginate(30);
        return view('admin.user.index', compact('user', 'page_title'));
    }

    public function transactionIndex()
    {
        $page_title = 'Transaction Log';
        $trans = Transaction::latest('id')->whereHas('user')->paginate(30);
        return view('admin.trans-log', compact('trans', 'page_title'));
    }

    public function usersActiveIndex()
    {
        $page_title = 'Active Users';
        $user = User::whereStatus(1)->latest('id')->paginate(30);
        return view('admin.user.index', compact('user', 'page_title'));
    }

    public function usersBanndedIndex()
    {
        $page_title = 'Banned Users';
        $user = User::whereStatus(0)->latest('id')->paginate(30);
        return view('admin.user.index', compact('user', 'page_title'));
    }

    public function userSearch(Request $request)
    {
        $page_title = 'User Group';
        $user = User::where('name', 'LIKE', "%{$request->username}%")->paginate();
        return view('admin.user.index', compact('user', 'page_title'));
    }

    public function userSearchEmail(Request $request)
    {
        $page_title = 'User Group';
        $user = User::where('email', 'LIKE', "%{$request->email}%")->paginate();
        return view('admin.user.index', compact('user', 'page_title'));
    }

    public function indexUserDetail($id)
    {
        $page_title = 'Users Details';
        $user = User::findOrFail($id);
        $wallets = Wallet::where('user_id', $id)->get();
        $rankings = Ranking::where('status', 1)->get();
        return view('admin.user.view', compact('user', 'wallets', 'page_title', 'rankings'));
    }

    function userRankUpdate(Request $request, $id)
    {
        $request->validate(['ranking' => 'required']);

        $user = User::findOrFail($id);

        $ranking = Ranking::findOrFail($request->ranking);

        $investedAmount = $user->investLog->sum('invest_amount');
        $depositAmount = $user->deposits->sum('amount');
        $totalRef = $user->refUser->count();

        if (!($investedAmount >= $ranking->minimum_invest)) {
            return redirect()->back()->with('alert', 'Not Enough amount invested');
        }

        if (!($depositAmount >= $ranking->minimum_deposit)) {
            return redirect()->back()->with('alert', 'Not Enough amount deposited');
        }

        if (!($totalRef >= $ranking->minimum_referral)) {
            return redirect()->back()->with('alert', 'Not Enough referred user');
        }

        if ($user->rankings->isNotEmpty()) {
            $user->rankings()->update('is_current', 0);
        }

        UserRanking::create([
            'user_id' => $user->id,
            'ranking_id' => $ranking->id,
            'bonus' => $ranking->bonus,
            'is_current' => 1,
        ]);

        $user->balance += $ranking->bonus;
        $user->save();

        return redirect()->back()->with('success', 'User Ranking update successfully');
    }

    public function activeUserUpdate(Request $request, $id)
    {
        $wallets = Wallet::findOrFail($id);
        Wallet::whereId($wallets->id)->update([
            'status' => 1,
        ]);
        return redirect()->back()->with('success', 'Received Coin Reject Success');
    }

    public function rejectUserUpdate(Request $request, $id)
    {
        $wallets = Wallet::findOrFail($id);
        Wallet::whereId($wallets->id)->update([
            'status' => 0,
        ]);
        return redirect()->back()->with('success', 'Received Coin Reject Success');
    }

    public function userUpdate(Request $request, $id)
    {
        $request->validate([
            'name' => 'required',
        ]);
        $user = User::findOrFail($id);
        $user->update($request->all());
        return redirect()->back()->with('success', 'Successfully Updated');
    }

    public function indexUserBalance($id)
    {
        $page_title = 'Users Balance';
        $user = User::findOrFail($id);
        return view('admin.user.balance', compact('user', 'page_title'));
    }

    public function indexBalanceUpdate(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'operation' => 'required',
        ]);
        $user = User::find($id);
        $general = General::first();
        if ($user instanceof User) {
            if ($request->operation == 1) {
                $new_balance = $user->balance + $request->amount;
                createTransaction('Balance added via admin', $request->amount, $user->balance, $new_balance, 1, $user->id);
                $user->balance = $new_balance;
                $user->update();
                if (!is_null($request->message)) {
                    $shortCodes = [
                        'amount' => $request->amount,
                        'currency' => $general->currency,
                        'post_balance' => $new_balance,
                        'post_message' => $request->message,
                    ];
                    @send_email($user, 'BAL_ADD', $shortCodes);
                }
                return back()->with('success', 'Balance Add Successful');
            } else {
                if ($user->balance >= $request->amount) {
                    $new_balance = $user->balance - $request->amount;
                    createTransaction('Balance deduct via admin', $request->amount, $user->balance, $new_balance, 1, $user->id);
                    $user->balance = $new_balance;
                    $user->update();
                    if (!is_null($request->message)) {
                        $shortCodes = [
                            'amount' => $request->amount,
                            'currency' => $general->currency,
                            'post_balance' => $new_balance,
                            'post_message' => $request->message,
                        ];
                        @send_email($user, 'BAL_SUB', $shortCodes);
                    }
                    return back()->with('success', 'Balance Subtract Successful');
                }
                return back()->with('alert', 'Insufficient Balance.');
            }
        }
        return back()->with('alert', 'User not found.');
    }

    public function userSendMail($id)
    {
        $page_title = 'Send Email Users';
        $user = User::findOrFail($id);
        return view('admin.user.user_mail', compact('user', 'page_title'));
    }

    public function userSendMailUser(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $subject = $request->subject;
        $message = $request->message;
        @send_email($user->email, $user->name, $subject, $message);
        return back()->with('success', 'Mail Send');
    }

    public function changePass()
    {
        $page_title = 'Profile';
        return view('admin.pass_change', compact('page_title'));
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email',
            'current_password' => 'nullable',
            'password' => 'nullable|confirmed',
        ]);
        $user = auth()->guard('admin')->user();

        if ($request->current_password) {
            if (Hash::check($request->current_password, Auth::guard('admin')->user()->password)) {
                $user->password = Hash::make($request->password);
            } else {
                return back()->with('alert', 'Old password not matched');
            }
        }

        $user->name = $request->name;
        $user->email = $request->email;

        $user->save();

        return back()->with('success', 'Profile update successful');
    }

    public function searchResult(Request $request)
    {
        $trans = Transaction::query();
        if (!is_null($request->trans_id)) {
            $trans->where('trans_id', 'LIKE', "%{$request->trans_id}%");
        }
        if (!is_null($request->user)) {
            $u = $request->user;
            $trans->whereHas('user', function ($q) use ($u) {
                $q->where('name', 'LIKE', "%{$u}%");
            });
        }
        if (!is_null($request->type)) {
            switch ($request->type) {
                case 'Invest':
                    $trans->where('status', 0);
                    break;
                case 'Deposit':
                    $trans->where('status', 1);
                    break;
                case 'Transfer':
                    $trans->where('status', 2);
                    break;
                case 'Income':
                    $trans->where('status', 4);
                    break;
                case 'Withdraw':
                    $trans->where('status', 3);
                    break;
                case 'Referral':
                    $trans->where('status', 5);
                    break;
                default:
                    $trans->whereIn('status', [0, 1, 2, 3, 4, 5]);
            }
        }
        $trans = $trans->latest('id')->paginate(50);
        $page_title = 'Transaction Log';
        return view('admin.trans-log', compact('trans', 'page_title'));
    }

    public function referralIndex()
    {
        $page_title = 'Manage Referral';
        $ref = Referral::all();
        $lastRef = Referral::orderBy('id', 'desc')->first();

        return view('admin.referral.index', compact('ref', 'lastRef', 'page_title'));
    }

    public function referralStore(Request $request)
    {
        if (count($request->percentage) == 0) {
            return back()->with('alert', 'Percentage field is required');
        }
        try {
            foreach ($request->percentage as $data) {
                if (!is_null($data)) {
                    if (!is_numeric($data)) {
                        return back()->with('alert', 'Please insert numeric value.');
                    }
                }
            }
            Referral::truncate();
            foreach ($request->percentage as $data) {
                Referral::create([
                    'percentage' => $data,
                ]);
            }
            return back()->with('success', 'Referral Percentage Commission generated.');
        } catch (\Exception $e) {
            return back()->with('success', 'Referral Percentage Commission generated.');
        }
    }
}

?>
