<?php

namespace App\Http\Controllers\Installer;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;

class InstallerController extends Controller
{
    function index()
    {
        $minimumPhpVersion = "8.2.0";

        $extensionLoaded = get_loaded_extensions();

        $requiredExtension = ['ctype', 'curl', 'dom', 'fileinfo', 'filter', 'hash', 'mbstring', 'openssl', 'pcre', 'PDO', 'session', 'tokenizer', 'xml', 'gd', 'pdo_mysql', 'bcmath'];

        $permissionFolders = ['./scripts/bootstrap/cache/', './scripts/storage/', './scripts/storage/app/', './scripts/storage/framework/', './scripts/storage/logs/'];

        foreach ($permissionFolders as $permission) {
            $permissionStatus = substr(sprintf('%o', fileperms($permission)), -4);
        }

        return view('installer.index', compact('extensionLoaded', 'requiredExtension', 'permissionFolders', 'minimumPhpVersion'));
    }

    function database(Request $request)
    {

        $request->validate([
            "db_name" => "required",
            "db_username" => "required",
            "db_password" => "nullable",
            "db_host" => "required",
            "db_port" => "required"
        ]);


        $host = $request->input('db_host');
        $database = $request->input('db_name');
        $username = $request->input('db_username');
        $password = $request->input('db_password');

        $conn = '';

        try {
            $conn = new \PDO("mysql:host=$host;port=$request->db_port;dbname=$database", $username, $password);
            $conn->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', "Could not connect to the database.  Please check your configuration");
        }
        
        
        try {
            $this->setEnvironmentValue('DB_DATABASE', $database);
            $this->setEnvironmentValue('DB_USERNAME', '"' . $username . '"');
             $this->setEnvironmentValue('DB_PASSWORD', '"' . $password . '"');
             $this->setEnvironmentValue('DB_HOST', $host);
             $this->setEnvironmentValue('DB_PORT', $request->db_port);
             
             Artisan::call('cache:clear');

        } catch (\Exception $e) {
            return redirect()->back()->with('error', "Connection Failed");
        }
        
        return redirect()->route('migrate');
        
    }
    function migrate()
    {
        return view('installer.migrate');
    }

    function migrateStore(Request $request)
    {
        
        $request->validate([
            'username' => 'required',
            'email' => 'required|email',
            'password' => 'required|confirmed',
        ]);
        
        
         // Disable foreign key checks to avoid issues
        Schema::disableForeignKeyConstraints();

        // Get all table names
        $tables = DB::select('SHOW TABLES');
        
        foreach ($tables as $table) {
            $tableName = array_values((array) $table)[0];

            Schema::drop($tableName);
        }

        // Enable foreign key checks again
        Schema::enableForeignKeyConstraints();
        
        $file = base_path("database.sql");

        if (File::exists($file)) {
            DB::unprepared(File::get($file));
        } else {
            return redirect()->route('installer')->with('error', 'No Database File Found');
        }
        
        

        Admin::truncate();

        Admin::create([
            'username' => $request->username,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'sct' => 'admin',
            'user_group_id' => 1,
            'status' => 1
        ]);

        return redirect()->route('success');

        
    }

    function success()
    {

        if (File::exists(base_path('init'))) {
            unlink(base_path('init'));
        }
        
        if(File::exists(app_path('Providers/Provider.php'))){
            File::copy(app_path('Providers/Provider.php'),app_path('Providers/AppServiceProvider.php'));
        }

        return view('installer.success');
    }


    function getPurchaseCode($code)
    {
        $personalToken = "YOUR_PERSONAL_TOKEN_HERE";

        // Surrounding whitespace can cause a 404 error, so trim it first
        $code = trim($code);

        // Make sure the code looks valid before sending it to Envato
        // This step is important - requests with incorrect formats can be blocked!
        if (!preg_match("/^([a-f0-9]{8})-(([a-f0-9]{4})-){3}([a-f0-9]{12})$/i", $code)) {
            return ['type'=> false, 'message' => "Invalid purchase code"];
        }

        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => "https://api.envato.com/v3/market/author/sale?code={$code}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 20,
            CURLOPT_HTTPHEADER => array(
                "Authorization: Bearer {$personalToken}",
                "User-Agent: Purchase code verification script"
            )
        ));

        $response = @curl_exec($ch);
        $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch) > 0) {
            return ['type'=> false, 'message' =>  "Failed to connect: " . curl_error($ch)];
        }

        switch ($responseCode) {
            case 404:
                return  ['type'=> false, 'message' =>  "Invalid purchase code"];
            case 403:
                return  ['type'=> false, 'message' =>  "The personal token is missing the required permission for this script"];
            case 401:
                return  ['type'=> false, 'message' =>  "The personal token is invalid or has been deleted"];
        }

        if ($responseCode !== 200) {
            return  ['type'=> false, 'message' =>  "Got status {$responseCode}, try again shortly"];
        }

        $body = @json_decode($response);

        if ($body === false && json_last_error() !== JSON_ERROR_NONE) {
            return  ['type'=> false, 'message' =>  "Error parsing response, try again"];
        }

        return  ['type'=> true, 'message' =>  $body];
    }


    function setEnvironmentValue($key, $value)
    {
        $path = base_path('.env');


        if (File::exists($path)) {
            // Get current env file content
            $env = File::get($path);


            // Replace the line with the new value
            if (strpos($env, "{$key}=") !== false) {
                $env = preg_replace("/^{$key}=.*/m", "{$key}={$value}", $env);
            } else {
                // If the key doesn't exist, add it
                $env .= "\n{$key}={$value}";
            }

            // Write the new env file content
            File::put($path, $env);
        }
    }
}
